<?php
session_start();

if (!isset($_SESSION['logged']) || $_SESSION['logged'] !== true) {
    header('Location: admin-login.php');
    exit;
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

$client = $_GET['client'] ?? 'ECF_VIGIER';

$dataRoot  = "/var/www/docnlink/data";
$enginsDir = $dataRoot . "/clients/$client/engins";

if (!is_dir($enginsDir)) {
    die("Dossier engins introuvable : $enginsDir");
}

$today = strtotime(date("Y-m-d"));

function analyse_prochain_entretien(?string $d): array {
    if (!$d) {
        return [
            'label' => 'Non planifié',
            'class' => 'badge-grey',
            'date'  => '—'
        ];
    }

    $ts = strtotime($d);
    if (!$ts) {
        return [
            'label' => 'Format invalide',
            'class' => 'badge-grey',
            'date'  => htmlspecialchars($d)
        ];
    }

    $today = strtotime(date("Y-m-d"));
    $diffDays = ($ts - $today) / 86400;

    if ($diffDays < 0) {
        $label = 'En retard';
        $class = 'badge-red';
    } elseif ($diffDays <= 30) {
        $label = 'Bientôt';
        $class = 'badge-orange';
    } else {
        $label = 'OK';
        $class = 'badge-green';
    }

    return [
        'label' => $label,
        'class' => $class,
        'date'  => date('d/m/y', $ts)
    ];
}

$rows = [];

foreach (glob($enginsDir . "/*.json") as $file) {
    $id   = basename($file, ".json");
    $data = json_decode(file_get_contents($file), true);
    if (!is_array($data)) continue;

    $marque = $data['nom']   ?? '';
    $type   = $data['type']  ?? '';
    $immat  = $data['immat'] ?? $id;

    $entretiens = $data['entretiens'] ?? [];

    $dernierDate    = null;
    $dernierType    = '';
    $prochainDate   = null;
    $prochainType   = '';

    if (is_array($entretiens) && !empty($entretiens)) {
        foreach ($entretiens as $e) {
            // Dernier entretien : max sur "date"
            if (!empty($e['date'])) {
                $ts = strtotime($e['date']);
                if ($ts && ($dernierDate === null || $ts > $dernierDate)) {
                    $dernierDate = $ts;
                    $dernierType = $e['type_entretien'] ?? '';
                }
            }

            // Prochain entretien : on prend la date la plus proche dans le futur
            if (!empty($e['prochain_entretien'])) {
                $ts2 = strtotime($e['prochain_entretien']);
                if ($ts2) {
                    if ($prochainDate === null || $ts2 < $prochainDate) {
                        $prochainDate = $ts2;
                        $prochainType = $e['type_entretien'] ?? '';
                    }
                }
            }
        }
    }

    $dernierDateStr  = $dernierDate  ? date('d/m/y', $dernierDate)  : '—';
    $prochainDateRaw = $prochainDate ? date('Y-m-d', $prochainDate) : null;

    $infoProchain = analyse_prochain_entretien($prochainDateRaw);

    $rows[] = [
        'id'              => $id,
        'immat'           => $immat,
        'marque'          => $marque,
        'type'            => $type,
        'dernier_date'    => $dernierDateStr,
        'dernier_type'    => $dernierType,
        'prochain'        => $infoProchain,
        'prochain_type'   => $prochainType,
    ];
}

// Tri par statut (rouge -> orange -> vert -> gris) puis par parc
usort($rows, function($a, $b) {
    $order = ['badge-red' => 0, 'badge-orange' => 1, 'badge-green' => 2, 'badge-grey' => 3];

    $ca = $order[$a['prochain']['class']] ?? 4;
    $cb = $order[$b['prochain']['class']] ?? 4;

    if ($ca === $cb) {
        return strcmp($a['immat'], $b['immat']);
    }
    return $ca <=> $cb;
});
?>
<!DOCTYPE html>
<html lang="fr">
<head>
<meta charset="UTF-8">
<title>Entretiens – Admin DocNLink</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
    :root {
        --primary:#00428B;
        --danger:#d9534f;
        --bg:#f3f3f3;
    }
    body {
        margin:0;
        font-family:Arial, sans-serif;
        background:var(--bg);
    }
    .topbar {
        background:var(--primary);
        color:white;
        padding:10px 15px;
        display:flex;
        justify-content:space-between;
        align-items:center;
        flex-wrap:wrap;
        gap:8px;
    }
    .topbar-title {
        font-weight:bold;
        font-size:18px;
    }
    .topbar-links a {
        color:white;
        text-decoration:none;
        margin-left:10px;
        font-size:14px;
    }
    .topbar-links a.active {
        font-weight:bold;
        text-decoration:underline;
    }
    .container {
        max-width:1100px;
        margin:15px auto;
        padding:0 10px;
    }
    .card {
        background:white;
        border-radius:10px;
        padding:15px;
        box-shadow:0 2px 6px rgba(0,0,0,0.15);
        margin-bottom:15px;
    }
    table {
        width:100%;
        border-collapse:collapse;
        font-size:14px;
    }
    th, td {
        padding:8px;
        border-bottom:1px solid #eee;
        text-align:left;
        vertical-align:top;
    }
    th {
        background:#f0f0f0;
        font-weight:bold;
    }
    tr:hover {
        background:#fafafa;
    }
    .badge {
        display:inline-block;
        padding:3px 6px;
        border-radius:999px;
        font-size:11px;
        font-weight:bold;
    }
    .badge-red    { background:#d9534f; color:white; }
    .badge-orange { background:#f0ad4e; color:white; }
    .badge-green  { background:#5cb85c; color:white; }
    .badge-grey   { background:#999; color:white; }

    .search-bar {
        margin-bottom:10px;
    }
    .search-input {
        width:100%;
        max-width:300px;
        padding:8px;
        border-radius:6px;
        border:1px solid #ccc;
        font-size:14px;
    }

    .btn {
        display:inline-block;
        padding:6px 8px;
        border-radius:6px;
        text-decoration:none;
        font-size:12px;
        border:none;
        cursor:pointer;
        background:var(--primary);
        color:white;
    }

    @media (max-width:800px) {
        table, thead, tbody, th, td, tr {
            display:block;
        }
        thead { display:none; }
        tr {
            margin-bottom:10px;
            border:1px solid #eee;
            border-radius:8px;
            padding:8px;
        }
        td {
            border-bottom:none;
            padding:4px 0;
        }
        td::before {
            content:attr(data-label) " : ";
            font-weight:bold;
        }
    }
</style>

<script>
document.addEventListener("DOMContentLoaded", function() {
    const input = document.getElementById("searchInput");
    const rows  = document.querySelectorAll("#entTable tbody tr");

    if (!input) return;

    input.addEventListener("input", function() {
        const q = this.value.toLowerCase().trim();

        rows.forEach(row => {
            const parc   = row.getAttribute("data-parc") || "";
            const marque = row.getAttribute("data-marque") || "";

            if (parc.toLowerCase().includes(q) || marque.toLowerCase().includes(q)) {
                row.style.display = "";
            } else {
                row.style.display = "none";
            }
        });
    });
});
</script>
</head>
<body>

<div class="topbar">
    <div class="topbar-title">
        🔧 Entretiens – DocNLink (<?php echo htmlspecialchars($client); ?>)
    </div>
    <div class="topbar-links">
        <a href="admin-panel.php?client=<?php echo urlencode($client); ?>">📋 Liste des engins</a>
        <a href="admin-documents-global.php?client=<?php echo urlencode($client); ?>">📑 Documents</a>
        <a href="admin-entretiens-global.php?client=<?php echo urlencode($client); ?>" class="active">🔧 Entretiens</a>
        <a href="admin-logout.php">🚪 Déconnexion</a>
    </div>
</div>

<div class="container">
    <div class="card">
        <h2>🔧 Tableau global des entretiens</h2>

        <div class="search-bar">
            <input
                type="text"
                id="searchInput"
                class="search-input"
                placeholder="Rechercher par numéro de parc ou marque…">
        </div>

        <?php if (empty($rows)): ?>
            <p>Aucun engin trouvé.</p>
        <?php else: ?>
            <table id="entTable">
                <thead>
                    <tr>
                        <th>Parc / ID</th>
                        <th>Marque</th>
                        <th>Type</th>
                        <th>Dernier entretien</th>
                        <th>Type</th>
                        <th>Prochain entretien</th>
                        <th>Statut</th>
                        <th>Type prévu</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($rows as $r): ?>
                    <tr
                        data-parc="<?php echo htmlspecialchars($r['immat']); ?>"
                        data-marque="<?php echo htmlspecialchars($r['marque']); ?>"
                    >
                        <td data-label="Parc / ID">
                            <?php echo htmlspecialchars($r['immat']); ?>
                            <br><small>(ID interne : <?php echo htmlspecialchars($r['id']); ?>)</small>
                        </td>
                        <td data-label="Marque"><?php echo htmlspecialchars($r['marque']); ?></td>
                        <td data-label="Type"><?php echo htmlspecialchars($r['type']); ?></td>

                        <td data-label="Dernier entretien">
                            <?php echo htmlspecialchars($r['dernier_date']); ?>
                        </td>
                        <td data-label="Type dernier">
                            <?php echo htmlspecialchars($r['dernier_type']); ?>
                        </td>

                        <td data-label="Prochain entretien">
                            <?php echo htmlspecialchars($r['prochain']['date']); ?>
                        </td>
                        <td data-label="Statut">
                            <span class="badge <?php echo $r['prochain']['class']; ?>">
                                <?php echo htmlspecialchars($r['prochain']['label']); ?>
                            </span>
                        </td>
                        <td data-label="Type prévu">
                            <?php echo htmlspecialchars($r['prochain_type']); ?>
                        </td>
                        <td data-label="Actions">
                            <a class="btn"
                               href="admin-entretien.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($r['id']); ?>">
                                ✏️ Détails
                            </a>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

</body>
</html>
