<?php
session_start();

// Sécurité : accès réservé
if (!isset($_SESSION['logged']) || $_SESSION['logged'] !== true) {
    header('Location: admin-login.php');
    exit;
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Client par défaut
$client = $_GET['client'] ?? 'ECF_VIGIER';

$dataRoot  = "/var/www/docnlink/data";
$enginsDir = $dataRoot . "/clients/$client/engins";

if (!is_dir($enginsDir)) {
    die("Dossier engins introuvable : $enginsDir");
}

// ========================
// Fonction d'analyse date doc (assurance, VGP, CT…)
// ========================
function analyse_date_doc(?string $d): array {
    if (!$d) {
        return [
            'label' => 'Non renseigné',
            'class' => 'badge-grey',
            'date'  => '—'
        ];
    }

    $ts = strtotime($d);
    if (!$ts) {
        return [
            'label' => 'Format invalide',
            'class' => 'badge-grey',
            'date'  => htmlspecialchars($d)
        ];
    }

    $today    = strtotime(date("Y-m-d"));
    $diffDays = ($ts - $today) / 86400;

    if ($diffDays < 0) {
        $label = 'ÉCHU';
        $class = 'badge-red';
    } elseif ($diffDays <= 30) {
        $label = 'Bientôt';
        $class = 'badge-orange';
    } else {
        $label = 'OK';
        $class = 'badge-green';
    }

    return [
        'label' => $label,
        'class' => $class,
        'date'  => date('d/m/y', $ts)
    ];
}

// ========================
// Chargement des engins + analyse docs & entretiens
// ========================
$engins = [];
$today  = strtotime(date("Y-m-d"));

// Alertes entretiens
$globalMaintRed    = []; // engins en retard
$globalMaintOrange = []; // engins bientôt à faire

// Alertes documents
$globalDocsRed    = [];  // engins avec docs expirés
$globalDocsOrange = [];  // engins docs bientôt périmés

$docFields = [
    'assurance' => 'Assurance',
    'vgp'       => 'VGP',
    'ct'        => 'Contrôle technique'
];

foreach (glob($enginsDir . "/*.json") as $file) {

    $id   = basename($file, ".json");
    $data = json_decode(file_get_contents($file), true);

    if (!is_array($data)) {
        continue;
    }

    $marque = $data['nom']   ?? '';
    $type   = $data['type']  ?? '';
    $immat  = $data['immat'] ?? $id;

    // -----------------------------------
    // 1) Analyse entretiens
    // -----------------------------------
    $etatEntretien   = 'GREEN'; // GREEN, ORANGE, RED
    $worstMaintTs    = null;   // date la plus "grave" (en retard ou proche)
    $entretiens      = $data['entretiens'] ?? [];

    if (is_array($entretiens) && !empty($entretiens)) {

        foreach ($entretiens as $e) {

            // Compatibilité anciennes / nouvelles clés
            $nextDateStr = '';
            if (!empty($e['prochain_entretien'])) {
                $nextDateStr = $e['prochain_entretien'];
            } elseif (!empty($e['prochain_entretien_date'])) {
                $nextDateStr = $e['prochain_entretien_date'];
            }

            if (!$nextDateStr) {
                continue;
            }

            $ts = strtotime($nextDateStr);
            if (!$ts) {
                continue;
            }

            $diffDays = ($ts - $today) / 86400;

            if ($diffDays < 0) {
                // En retard
                $etatEntretien = 'RED';
                if ($worstMaintTs === null || $ts < $worstMaintTs) {
                    $worstMaintTs = $ts;
                }
                // On peut continuer pour voir s'il y a pire, mais RED est déjà le max
            } elseif ($diffDays <= 30 && $etatEntretien !== 'RED') {
                // Bientôt à faire
                if ($etatEntretien !== 'ORANGE') {
                    $etatEntretien = 'ORANGE';
                }
                if ($worstMaintTs === null || $ts < $worstMaintTs) {
                    $worstMaintTs = $ts;
                }
            }
        }
    }

    if ($etatEntretien === 'RED') {
        $globalMaintRed[] = [
            'id'    => $id,
            'immat' => $immat,
            'date'  => $worstMaintTs ? date('d/m/y', $worstMaintTs) : '—'
        ];
    } elseif ($etatEntretien === 'ORANGE') {
        $globalMaintOrange[] = [
            'id'    => $id,
            'immat' => $immat,
            'date'  => $worstMaintTs ? date('d/m/y', $worstMaintTs) : '—'
        ];
    }

    // -----------------------------------
    // 2) Analyse documents (assurance, VGP, CT)
    // -----------------------------------
    $etatDocs   = 'GREEN'; // GREEN, ORANGE, RED
    $docsIssues = [];

    foreach ($docFields as $key => $label) {
        $val = $data['validite'][$key] ?? null;
        if (!$val) {
            continue;
        }

        $info = analyse_date_doc($val);

        if ($info['label'] === 'ÉCHU') {
            $etatDocs = 'RED';
            $docsIssues[] = [
                'doc'  => $label,
                'date' => $info['date']
            ];
        } elseif ($info['label'] === 'Bientôt' && $etatDocs !== 'RED') {
            if ($etatDocs !== 'ORANGE') {
                $etatDocs = 'ORANGE';
            }
            $docsIssues[] = [
                'doc'  => $label,
                'date' => $info['date']
            ];
        }
    }

    if ($etatDocs === 'RED' && !empty($docsIssues)) {
        $globalDocsRed[] = [
            'id'     => $id,
            'immat'  => $immat,
            'issues' => $docsIssues
        ];
    } elseif ($etatDocs === 'ORANGE' && !empty($docsIssues)) {
        $globalDocsOrange[] = [
            'id'     => $id,
            'immat'  => $immat,
            'issues' => $docsIssues
        ];
    }

    $engins[] = [
        'id'             => $id,
        'immat'          => $immat,
        'marque'         => $marque,
        'type'           => $type,
        'etat_entretien' => $etatEntretien,
        'etat_docs'      => $etatDocs
    ];
}

// Tri : par ID interne
usort($engins, function($a, $b) {
    return strcmp($a['id'], $b['id']);
});
?>
<!DOCTYPE html>
<html lang="fr">
<head>
<meta charset="UTF-8">
<title>Admin DocNLink</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
    :root {
        --primary:#00428B;
        --danger:#d9534f;
        --bg:#f3f3f3;
    }
    body {
        margin:0;
        font-family:Arial, sans-serif;
        background:var(--bg);
    }
    .topbar {
        background:var(--primary);
        color:white;
        padding:10px 15px;
        display:flex;
        justify-content:space-between;
        align-items:center;
    }
    .topbar-title {
        font-weight:bold;
        font-size:18px;
    }
    .topbar-links a {
        color:white;
        text-decoration:none;
        margin-left:10px;
        font-size:14px;
    }
    .container {
        max-width:1000px;
        margin:15px auto;
        padding:0 10px;
    }
    .card {
        background:white;
        border-radius:10px;
        padding:15px;
        box-shadow:0 2px 6px rgba(0,0,0,0.15);
        margin-bottom:15px;
    }
    .btn {
        display:inline-block;
        padding:8px 10px;
        border-radius:6px;
        text-decoration:none;
        font-size:13px;
        border:none;
        cursor:pointer;
    }
    .btn-primary { background:var(--primary); color:white; }
    .btn-qr      { background:#008000; color:white; }
    .btn-doc     { background:#555;   color:white; }
    .btn-ent     { background:#f0ad4e;color:white; }
    .btn-del     { background:#d9534f;color:white; }
    .btn-small   { font-size:12px; padding:6px 8px; }

    table {
        width:100%;
        border-collapse:collapse;
        font-size:14px;
    }
    th, td {
        padding:8px;
        border-bottom:1px solid #eee;
        text-align:left;
        vertical-align:top;
    }
    th {
        background:#f0f0f0;
        font-weight:bold;
    }
    tr:hover {
        background:#fafafa;
    }
    .search-bar {
        margin-bottom:10px;
    }
    .search-input {
        width:100%;
        max-width:300px;
        padding:8px;
        border-radius:6px;
        border:1px solid #ccc;
        font-size:14px;
    }
    .badge {
        display:inline-block;
        padding:3px 6px;
        border-radius:999px;
        font-size:11px;
        font-weight:bold;
    }
    .badge-red {
        background:#d9534f;
        color:white;
    }
    .badge-orange {
        background:#f0ad4e;
        color:white;
    }
    .badge-green {
        background:#5cb85c;
        color:white;
    }
    .badge-grey {
        background:#999;
        color:white;
    }
    .alert-box {
        padding:12px;
        border-radius:8px;
        margin-bottom:15px;
        color:white;
        font-weight:bold;
        font-size:14px;
    }
    .alert-red   { background:#d9534f; }
    .alert-orange{ background:#f0ad4e; }
    .alert-box ul {
        margin:6px 0 0 18px;
        padding:0;
        font-weight:normal;
        font-size:13px;
    }

    @media (max-width:700px) {
        table, thead, tbody, th, td, tr {
            display:block;
        }
        thead { display:none; }
        tr {
            margin-bottom:10px;
            border:1px solid #eee;
            border-radius:8px;
            padding:8px;
        }
        td {
            border-bottom:none;
            padding:4px 0;
        }
        td::before {
            content:attr(data-label) " : ";
            font-weight:bold;
        }
    }
</style>

<script>
// Filtre de recherche sur le numéro de parc / marque
document.addEventListener("DOMContentLoaded", function() {
    const input = document.getElementById("searchInput");
    const rows  = document.querySelectorAll("#enginsTable tbody tr");

    if (!input) return;

    input.addEventListener("input", function() {
        const q = this.value.toLowerCase().trim();

        rows.forEach(row => {
            const parc   = row.getAttribute("data-parc") || "";
            const marque = row.getAttribute("data-marque") || "";

            if (parc.toLowerCase().includes(q) || marque.toLowerCase().includes(q)) {
                row.style.display = "";
            } else {
                row.style.display = "none";
            }
        });
    });
});
</script>
</head>
<body>

<div class="topbar">
    <div class="topbar-title">
        🔧 DocNLink – Administration (<?php echo htmlspecialchars($client); ?>)
    </div>
    <div class="topbar-links">
        <a href="admin-create-engin.php?client=<?php echo urlencode($client); ?>">➕ Créer un engin</a>
        <a href="admin-logout.php">🚪 Déconnexion</a>
    </div>
</div>

<div class="container">

    <!-- Alertes DOCUMENTS -->
    <?php if (!empty($globalDocsRed)): ?>
        <div class="alert-box alert-red">
            ❗ Documents expirés sur <?php echo count($globalDocsRed); ?> engin(s) :
            <ul>
                <?php foreach ($globalDocsRed as $e): ?>
                    <li>
                        <?php echo htmlspecialchars($e['immat']); ?>
                        (ID <?php echo htmlspecialchars($e['id']); ?>) :
                        <?php
                        $labels = array_map(function($d){ return $d['doc'] . " (" . $d['date'] . ")"; }, $e['issues']);
                        echo htmlspecialchars(implode(", ", $labels));
                        ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php elseif (!empty($globalDocsOrange)): ?>
        <div class="alert-box alert-orange">
            ⚠️ Documents bientôt expirés sur <?php echo count($globalDocsOrange); ?> engin(s) :
            <ul>
                <?php foreach ($globalDocsOrange as $e): ?>
                    <li>
                        <?php echo htmlspecialchars($e['immat']); ?>
                        (ID <?php echo htmlspecialchars($e['id']); ?>) :
                        <?php
                        $labels = array_map(function($d){ return $d['doc'] . " (" . $d['date'] . ")"; }, $e['issues']);
                        echo htmlspecialchars(implode(", ", $labels));
                        ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <!-- Alertes ENTRETIENS -->
    <?php if (!empty($globalMaintRed)): ?>
        <div class="alert-box alert-red">
            ⚠️ Entretien en retard sur <?php echo count($globalMaintRed); ?> engin(s) :
            <ul>
                <?php foreach ($globalMaintRed as $e): ?>
                    <li>
                        <?php echo htmlspecialchars($e['immat']); ?>
                        (ID <?php echo htmlspecialchars($e['id']); ?>)
                        – échéance le <?php echo htmlspecialchars($e['date']); ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php elseif (!empty($globalMaintOrange)): ?>
        <div class="alert-box alert-orange">
            ⚠️ Entretien à prévoir dans les 30 jours sur <?php echo count($globalMaintOrange); ?> engin(s) :
            <ul>
                <?php foreach ($globalMaintOrange as $e): ?>
                    <li>
                        <?php echo htmlspecialchars($e['immat']); ?>
                        (ID <?php echo htmlspecialchars($e['id']); ?>)
                        – prévu le <?php echo htmlspecialchars($e['date']); ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <div class="card">
        <h2>📋 Liste des engins</h2>

        <div class="search-bar">
            <input
                type="text"
                id="searchInput"
                class="search-input"
                placeholder="Rechercher par numéro de parc ou marque…">
        </div>

        <?php if (empty($engins)): ?>
            <p>Aucun engin trouvé pour ce client.</p>
        <?php else: ?>
            <table id="enginsTable">
                <thead>
                    <tr>
                        <th>Parc / ID</th>
                        <th>Marque</th>
                        <th>Type</th>
                        <th>État docs</th>
                        <th>État entretien</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($engins as $e): ?>
                    <tr
                        data-parc="<?php echo htmlspecialchars($e['immat']); ?>"
                        data-marque="<?php echo htmlspecialchars($e['marque']); ?>"
                    >
                        <td data-label="Parc / ID">
                            <?php echo htmlspecialchars($e['immat']); ?>
                            <br><small>(ID interne : <?php echo htmlspecialchars($e['id']); ?>)</small>
                        </td>
                        <td data-label="Marque"><?php echo htmlspecialchars($e['marque']); ?></td>
                        <td data-label="Type"><?php echo htmlspecialchars($e['type']); ?></td>

                        <td data-label="État docs">
                            <?php if ($e['etat_docs'] === 'RED'): ?>
                                <span class="badge badge-red">🟥 Docs</span>
                            <?php elseif ($e['etat_docs'] === 'ORANGE'): ?>
                                <span class="badge badge-orange">🟧 Docs</span>
                            <?php else: ?>
                                <span class="badge badge-green">🟩 Docs</span>
                            <?php endif; ?>
                        </td>

                        <td data-label="État entretien">
                            <?php if ($e['etat_entretien'] === 'RED'): ?>
                                <span class="badge badge-red">🟥 Retard</span>
                            <?php elseif ($e['etat_entretien'] === 'ORANGE'): ?>
                                <span class="badge badge-orange">🟧 Bientôt</span>
                            <?php else: ?>
                                <span class="badge badge-green">🟩 OK</span>
                            <?php endif; ?>
                        </td>

                        <td data-label="Actions">
                            <a class="btn btn-primary btn-small"
                               href="fiche-engin.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($e['id']); ?>">
                                📄 Fiche
                            </a>
                            <a class="btn btn-doc btn-small"
                               href="admin-documents.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($e['id']); ?>">
                                📁 Docs
                            </a>
                            <a class="btn btn-ent btn-small"
                               href="admin-entretien.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($e['id']); ?>">
                                🔧 Entretiens
                            </a>
                            <a class="btn btn-qr btn-small"
                               href="admin-print-qr.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($e['id']); ?>"
                               target="_blank">
                                🖨️ QR
                            </a>

                            <!-- Suppression engin (mot de passe) -->
                            <form method="POST" action="admin-delete-engin.php"
                                  style="margin-top:6px;"
                                  onsubmit="return confirm('Supprimer définitivement cet engin (JSON + docs + photos) ?');">
                                <input type="hidden" name="client" value="<?php echo htmlspecialchars($client); ?>">
                                <input type="hidden" name="id" value="<?php echo htmlspecialchars($e['id']); ?>">
                                <input type="password" name="password" placeholder="Mot de passe" required
                                       style="width:100%; padding:4px; font-size:11px; margin-bottom:4px;">
                                <button type="submit" class="btn btn-del btn-small">🗑 Supprimer</button>
                            </form>

                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

</div>

</body>
</html>
