<?php
session_start();

error_reporting(E_ALL);
ini_set('display_errors', 1);

// ========================
// Chargement & sécurité
// ========================
$client = isset($_GET['client']) ? $_GET['client'] : 'ECF_VIGIER';
$id     = isset($_GET['id']) ? $_GET['id'] : '';

if ($id === '') {
    http_response_code(400);
    die("ID d'engin manquant.");
}

$basePath   = "/var/www/docnlink/data";
$jsonPath   = "$basePath/clients/$client/engins/$id.json";
$docsDir    = "$basePath/documents/$client/$id";
$docsUrlDir = "https://docnlink.fr/data/documents/$client/$id";

if (!file_exists($jsonPath)) {
    http_response_code(404);
    die("Fiche engin introuvable : $jsonPath");
}

$data = json_decode(file_get_contents($jsonPath), true);
if (!is_array($data)) {
    die("Erreur JSON engin.");
}

// ========================
// Fonctions utilitaires
// ========================
function analyse_date_doc(?string $d): array {
    if (!$d) {
        return [
            'label' => 'Non renseigné',
            'class' => 'badge-grey',
            'date'  => '—'
        ];
    }

    $ts = strtotime($d);
    if (!$ts) {
        return [
            'label' => 'Format invalide',
            'class' => 'badge-grey',
            'date'  => htmlspecialchars($d)
        ];
    }

    $today    = strtotime(date("Y-m-d"));
    $diffDays = ($ts - $today) / 86400;

    if ($diffDays < 0) {
        $label = 'ÉCHU';
        $class = 'badge-red';
    } elseif ($diffDays <= 30) {
        $label = 'Bientôt';
        $class = 'badge-orange';
    } else {
        $label = 'OK';
        $class = 'badge-green';
    }

    return [
        'label' => $label,
        'class' => $class,
        'date'  => date('d/m/y', $ts)
    ];
}

// Pour la petite mention dans la carte "Dates de validité"
function badgeDate($d) {
    if (!$d) return "";
    $ts = strtotime($d);
    if (!$ts) return "";
    $today = strtotime(date("Y-m-d"));
    if ($ts < $today) {
        return " <span style='color:red;font-size:12px;'> (EXPIRÉ)</span>";
    }
    return " <span style='color:green;font-size:12px;'> (" . date('d/m/y', $ts) . ")</span>";
}

// ========================
// Récupération des données
// ========================
$logo        = $data['logo'] ?? "";
$nomEngin    = $data['nom'] ?? $id;
$typeEngin   = $data['type'] ?? "";
$numeroSerie = $data['numero_serie'] ?? "";
$immat       = $data['immat'] ?? $id;
$annee       = $data['annee'] ?? "";
$poids       = $data['poids'] ?? "";

$photoUrl = (!empty($data['photo']))
    ? $data['photo']
    : "https://via.placeholder.com/800x400?text=Photo+non+disponible";

// Dates docs
$vgp       = $data['validite']['vgp'] ?? "";
$assurance = $data['validite']['assurance'] ?? "";

// Analyse pour le tableau doc
$infoVgp       = analyse_date_doc($vgp);
$infoAssurance = analyse_date_doc($assurance);

// ========================
// Documents dynamiques
// ========================
$docs = [];
if (is_dir($docsDir)) {
    foreach (glob($docsDir . "/*") as $file) {
        if (is_file($file)) {
            $base     = basename($file);
            $nameOnly = pathinfo($base, PATHINFO_FILENAME);
            $label    = str_replace(['_', '-'], ' ', $nameOnly);
            $label    = mb_convert_case($label, MB_CASE_TITLE);

            $docs[] = [
                "label" => $label,
                "url"   => $docsUrlDir . "/" . rawurlencode($base)
            ];
        }
    }
}

// ========================
// ENTRETIENS : lecture + ajout éventuel (utilisateur QR)
// ========================
$rawEntretiens = $data['entretiens'] ?? [];
$entretiens    = [];
$formError     = "";

// Normalisation lecture (compatibilité ancienne structure)
if (is_array($rawEntretiens)) {
    foreach ($rawEntretiens as $e) {
        $ent = [];
        $ent['date']    = $e['date'] ?? '';
        $ent['nom']     = $e['nom'] ?? '';
        $ent['garage_ou_chauffeur'] =
            $e['garage_ou_chauffeur'] ??
            ($e['garage'] ?? '');
        $ent['compteur'] = $e['compteur'] ?? '';
        $ent['type_entretien'] =
            $e['type_entretien'] ??
            ($e['type'] ?? '');
        $ent['commentaire'] = $e['commentaire'] ?? '';
        $ent['prochain_entretien'] =
            $e['prochain_entretien'] ??
            ($e['prochain_entretien_date'] ?? '');
        $ent['commentaire_prochain'] =
            $e['commentaire_prochain'] ??
            ($e['prochain_entretien_commentaire'] ?? '');
        $ent['compteur_prochain'] =
            $e['compteur_prochain'] ??
            ($e['prochain_entretien_compteur'] ?? '');

        $entretiens[] = $ent;
    }
}

// Traitement formulaire ajout entretien (utilisateur QR)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_entretien'])) {

    $userCap   = trim($_POST['captcha'] ?? '');
    $expected  = $_SESSION['captcha_sum'] ?? null;

    if ($expected === null || $userCap === '' || intval($userCap) !== intval($expected)) {
        $formError = "Captcha incorrect. Merci de recommencer.";
    } else {
        $ent = [
            'date'                 => trim($_POST['date'] ?? ''),
            'nom'                  => trim($_POST['nom'] ?? ''),
            'garage_ou_chauffeur'  => trim($_POST['garage_ou_chauffeur'] ?? ''),
            'compteur'             => trim($_POST['compteur'] ?? ''),
            'type_entretien'       => trim($_POST['type_entretien'] ?? ''),
            'commentaire'          => trim($_POST['commentaire'] ?? ''),
            'prochain_entretien'   => trim($_POST['prochain_entretien'] ?? ''),
            'commentaire_prochain' => trim($_POST['commentaire_prochain'] ?? ''),
            'compteur_prochain'    => trim($_POST['compteur_prochain'] ?? '')
        ];

        if ($ent['date'] === '' || $ent['type_entretien'] === '') {
            $formError = "La date et le type d'entretien sont obligatoires.";
        } else {
            // On pousse dans la structure JSON "officielle"
            if (!isset($data['entretiens']) || !is_array($data['entretiens'])) {
                $data['entretiens'] = [];
            }

            $data['entretiens'][] = [
                'date'                   => $ent['date'],
                'nom'                    => $ent['nom'],
                'garage_ou_chauffeur'    => $ent['garage_ou_chauffeur'],
                'compteur'               => $ent['compteur'],
                'type_entretien'         => $ent['type_entretien'],
                'commentaire'            => $ent['commentaire'],
                'prochain_entretien'     => $ent['prochain_entretien'],
                'commentaire_prochain'   => $ent['commentaire_prochain'],
                'compteur_prochain'      => $ent['compteur_prochain']
            ];

            if (file_put_contents($jsonPath, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) === false) {
                $formError = "Erreur lors de l'enregistrement de l'entretien.";
            } else {
                header("Location: fiche-engin.php?client=" . urlencode($client) . "&id=" . urlencode($id));
                exit;
            }
        }
    }
}

// Rechargement (au cas où)
$data         = json_decode(file_get_contents($jsonPath), true);
$rawEntretiens = $data['entretiens'] ?? [];
$entretiens    = [];

if (is_array($rawEntretiens)) {
    foreach ($rawEntretiens as $e) {
        $ent = [];
        $ent['date']    = $e['date'] ?? '';
        $ent['nom']     = $e['nom'] ?? '';
        $ent['garage_ou_chauffeur'] =
            $e['garage_ou_chauffeur'] ??
            ($e['garage'] ?? '');
        $ent['compteur'] = $e['compteur'] ?? '';
        $ent['type_entretien'] =
            $e['type_entretien'] ??
            ($e['type'] ?? '');
        $ent['commentaire'] = $e['commentaire'] ?? '';
        $ent['prochain_entretien'] =
            $e['prochain_entretien'] ??
            ($e['prochain_entretien_date'] ?? '');
        $ent['commentaire_prochain'] =
            $e['commentaire_prochain'] ??
            ($e['prochain_entretien_commentaire'] ?? '');
        $ent['compteur_prochain'] =
            $e['compteur_prochain'] ??
            ($e['prochain_entretien_compteur'] ?? '');

        $entretiens[] = $ent;
    }

    // Tri du plus récent au plus ancien
    usort($entretiens, function($a, $b) {
        $ta = !empty($a['date']) ? strtotime($a['date']) : 0;
        $tb = !empty($b['date']) ? strtotime($b['date']) : 0;
        return $tb <=> $ta;
    });
}

// Nouveau captcha
$a = rand(1, 9);
$b = rand(1, 9);
$_SESSION['captcha_sum'] = $a + $b;

// ========================
// Actions rapides (panne)
// ========================
$enginID = $immat ?: $id;

$mailTo      = "p.aubert@ecf-vigier.com";
$mailSubject = "Déclaration de panne – Engin $enginID";
$mailBody    = "Déclaration de panne\nEngin : $enginID\nCompteur d'heure / km :\nDate :\n\nMerci d’indiquer la panne et ajouter une photo si possible.";
$mailUrl     = "mailto:$mailTo?subject=" . rawurlencode($mailSubject) . "&body=" . rawurlencode($mailBody);

$waNumber = "33679505890";
$waMsg    = "Déclaration de panne\nEngin: $enginID\nCompteur d'heure / km:\nDate:\nMerci d’indiquer la panne et ajouter une photo si possible.";
$waUrl    = "https://wa.me/$waNumber?text=" . rawurlencode($waMsg);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
<meta charset="UTF-8">
<title>Fiche Engin - <?php echo htmlspecialchars($nomEngin); ?></title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
    body { margin:0; padding:0; background:#f3f3f3; font-family:Arial, sans-serif; }
    .header {
        background:#00428B; color:white; padding:10px 15px;
        display:flex; justify-content:space-between; align-items:center;
    }
    .header img { max-height:50px; }
    .container { padding:10px; max-width:900px; margin:auto; }
    .card {
        background:white; border-radius:8px;
        box-shadow:0 2px 6px rgba(0,0,0,0.15);
        padding:15px; margin-bottom:15px;
    }
    h1 { margin:0 0 8px 0; font-size:22px; }
    h2 {
        font-size:18px; border-bottom:1px solid #eee;
        margin:0 0 10px 0; padding-bottom:5px;
    }
    .infos-grid {
        display:grid; grid-template-columns:repeat(2,1fr);
        gap:8px; font-size:14px;
    }
    .label { font-weight:bold; }
    .photo img { width:100%; border-radius:8px; }
    ul { list-style:none; padding:0; margin:0; }
    .btn {
        display:block; background:#00428B; color:white;
        padding:10px; margin-bottom:8px;
        border-radius:6px; text-decoration:none; font-size:15px;
        text-align:center;
    }
    .btn-whatsapp { background:#25D366; }
    .btn-danger   { background:#d9534f; }
    .btn-grey     { background:#555; }
    .btn-small-inline {
        display:inline-block; margin-right:5px; padding:6px 10px; font-size:13px;
    }

    table.docs-table {
        width:100%; border-collapse:collapse; font-size:14px;
    }
    table.docs-table th, table.docs-table td {
        padding:6px 8px;
        border-bottom:1px solid #eee;
        text-align:left;
    }
    table.docs-table th { background:#f7f7f7; }

    .badge-doc {
        display:inline-block;
        padding:2px 8px;
        border-radius:999px;
        font-size:11px;
        font-weight:bold;
    }
    .badge-red    { background:#d9534f; color:white; }
    .badge-orange { background:#f0ad4e; color:white; }
    .badge-green  { background:#5cb85c; color:white; }
    .badge-grey   { background:#999; color:white; }

    table.entretien-table {
        width:100%; border-collapse:collapse; font-size:13px;
    }
    table.entretien-table th, table.entretien-table td {
        padding:6px 5px;
        border-bottom:1px solid #eee;
        vertical-align:top;
    }
    table.entretien-table th {
        background:#f7f7f7;
        position:sticky;
        top:0;
    }

    input[type="text"], input[type="date"], textarea {
        width:100%; padding:6px; border-radius:5px; border:1px solid #ccc;
        font-size:13px; box-sizing:border-box;
    }
    textarea { resize:vertical; }

    .msg-error   { color:#d9534f; font-weight:bold; margin-bottom:8px; }

    @media (max-width:700px) {
        .infos-grid { grid-template-columns:1fr; }
    }
</style>
</head>

<body>

<!-- EN-TÊTE -->
<div class="header">
    <div>
        <?php if ($logo): ?>
            <img src="<?php echo htmlspecialchars($logo); ?>" alt="Logo">
        <?php endif; ?>
    </div>
    <div><?php echo htmlspecialchars(str_replace('_',' ', $client)); ?></div>
</div>

<div class="container">

    <!-- INFOS -->
    <div class="card">
        <h1><?php echo htmlspecialchars($nomEngin); ?></h1>
        <div class="infos-grid">
            <div><span class="label">Type :</span> <?php echo htmlspecialchars($typeEngin); ?></div>
            <div><span class="label">N° Série :</span> <?php echo htmlspecialchars($numeroSerie); ?></div>
            <div><span class="label">Parc / Immat :</span> <?php echo htmlspecialchars($immat); ?></div>
            <div><span class="label">Année :</span> <?php echo htmlspecialchars($annee); ?></div>
            <div><span class="label">Poids :</span> <?php echo htmlspecialchars($poids); ?></div>
        </div>

        <!-- Boutons admin rapides -->
        <div style="margin-top:10px;">
            <a class="btn btn-grey btn-small-inline"
               href="admin-documents.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($id); ?>">
                ⚙️ Gérer les documents (admin)
            </a>
            <a class="btn btn-grey btn-small-inline"
               href="admin-entretien.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($id); ?>">
                🔧 Gérer les entretiens (admin)
            </a>
            <a class="btn btn-grey btn-small-inline"
               href="admin-print-qr.php?client=<?php echo urlencode($client); ?>&id=<?php echo urlencode($id); ?>"
               target="_blank">
                🖨️ QR (admin)
            </a>
        </div>
    </div>

    <!-- PHOTO -->
    <div class="card">
        <div class="photo">
            <img src="<?php echo htmlspecialchars($photoUrl); ?>" alt="Photo engin">
        </div>
    </div>

    <!-- VALIDITÉ (carte simple) -->
    <div class="card">
        <h2>Dates de validité</h2>
        <p><b>Assurance :</b> <?php echo $assurance ? htmlspecialchars(date('d/m/y', strtotime($assurance))) : "—"; ?> <?php echo badgeDate($assurance); ?></p>
        <p><b>VGP :</b> <?php echo $vgp ? htmlspecialchars(date('d/m/y', strtotime($vgp))) : "—"; ?> <?php echo badgeDate($vgp); ?></p>
    </div>

    <!-- TABLEAU DOCUMENTS -->
    <div class="card">
        <h2>📑 Suivi des documents</h2>
        <table class="docs-table">
            <thead>
                <tr>
                    <th>Document</th>
                    <th>Date</th>
                    <th>Statut</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>Assurance</td>
                    <td><?php echo htmlspecialchars($infoAssurance['date']); ?></td>
                    <td><span class="badge-doc <?php echo $infoAssurance['class']; ?>">
                        <?php echo htmlspecialchars($infoAssurance['label']); ?>
                    </span></td>
                </tr>
                <tr>
                    <td>VGP</td>
                    <td><?php echo htmlspecialchars($infoVgp['date']); ?></td>
                    <td><span class="badge-doc <?php echo $infoVgp['class']; ?>">
                        <?php echo htmlspecialchars($infoVgp['label']); ?>
                    </span></td>
                </tr>
            </tbody>
        </table>
    </div>

    <!-- DOCUMENTS DYNAMIQUES -->
    <div class="card">
        <h2>📂 Documents disponibles</h2>
        <?php if (empty($docs)): ?>
            <p style="color:#777;">Aucun document trouvé dans le dossier.</p>
        <?php else: ?>
            <ul>
                <?php foreach ($docs as $d): ?>
                    <li style="margin-bottom:6px;">
                        <a class="btn" href="<?php echo $d['url']; ?>" target="_blank">
                            📄 <?php echo htmlspecialchars($d['label']); ?>
                        </a>
                    </li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
    </div>

    <!-- HISTORIQUE ENTRETIENS -->
    <div class="card">
        <h2>🔧 Historique des entretiens</h2>

        <?php if (empty($entretiens)): ?>
            <p style="color:#777;">Aucun entretien enregistré pour le moment.</p>
        <?php else: ?>
            <div style="max-height:260px; overflow:auto;">
                <table class="entretien-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Nom</th>
                            <th>Garage / Chauffeur</th>
                            <th>Compteur (h / km)</th>
                            <th>Type d'entretien</th>
                            <th>Commentaire</th>
                            <th>Prochain entretien</th>
                            <th>Commentaire prochain</th>
                            <th>Compteur prochain</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($entretiens as $e): ?>
                            <tr>
                                <td><?php echo !empty($e['date']) ? htmlspecialchars(date('d/m/y', strtotime($e['date']))) : '—'; ?></td>
                                <td><?php echo htmlspecialchars($e['nom'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($e['garage_ou_chauffeur'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($e['compteur'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($e['type_entretien'] ?? ''); ?></td>
                                <td><?php echo nl2br(htmlspecialchars($e['commentaire'] ?? '')); ?></td>
                                <td><?php
                                    if (!empty($e['prochain_entretien'])) {
                                        echo htmlspecialchars(date('d/m/y', strtotime($e['prochain_entretien'])));
                                    } else {
                                        echo '—';
                                    }
                                ?></td>
                                <td><?php echo nl2br(htmlspecialchars($e['commentaire_prochain'] ?? '')); ?></td>
                                <td><?php echo htmlspecialchars($e['compteur_prochain'] ?? ''); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <!-- FORMULAIRE AJOUT ENTRETIEN (utilisateur QR) -->
    <div class="card">
        <h2>➕ Ajouter un entretien</h2>

        <?php if ($formError): ?>
            <div class="msg-error"><?php echo htmlspecialchars($formError); ?></div>
        <?php endif; ?>

        <form method="POST">
            <input type="hidden" name="add_entretien" value="1">

            <p>
                <label>Date :</label><br>
                <input type="date" name="date" required>
            </p>
            <p>
                <label>Nom :</label><br>
                <input type="text" name="nom" placeholder="Nom de la personne">
            </p>
            <p>
                <label>Garage ou Chauffeur :</label><br>
                <input type="text" name="garage_ou_chauffeur" placeholder="Garage ou chauffeur">
            </p>
            <p>
                <label>Compteur (heures ou kilomètres) :</label><br>
                <input type="text" name="compteur" placeholder="Ex : 3 250 h ou 125 000 km">
            </p>
            <p>
                <label>Type d'entretien :</label><br>
                <input type="text" name="type_entretien" placeholder="Vidange, révision, freinage..." required>
            </p>
            <p>
                <label>Commentaire :</label><br>
                <textarea name="commentaire" rows="3"></textarea>
            </p>

            <hr>

            <p>
                <label>Prochain entretien (date) :</label><br>
                <input type="date" name="prochain_entretien">
            </p>
            <p>
                <label>Compteur prévu (heures / km) :</label><br>
                <input type="text" name="compteur_prochain">
            </p>
            <p>
                <label>Commentaire prochain entretien :</label><br>
                <textarea name="commentaire_prochain" rows="2"></textarea>
            </p>

            <p>
                <label>Vérification (anti-robot) :</label><br>
                <b><?php echo $a . " + " . $b; ?> = ?</b><br>
                <input type="number" name="captcha" required style="width:100px; margin-top:4px;">
            </p>

            <button class="btn" type="submit">💾 Enregistrer l'entretien</button>
        </form>
    </div>

    <!-- ACTIONS RAPIDES (panne) -->
    <div class="card">
        <h2>Actions rapides</h2>
        <ul>
            <li><a class="btn" href="tel:+33679505890">📞 Appeler Maintenance</a></li>
            <li><a class="btn btn-danger" href="<?php echo $mailUrl; ?>">🚨 Déclarer une panne (Mail)</a></li>
            <li><a class="btn btn-whatsapp" href="<?php echo $waUrl; ?>" target="_blank">🟢 Déclarer une panne (WhatsApp)</a></li>
        </ul>
    </div>

</div>

</body>
</html>
